#include "Evenement.h"
#include "Temps.h"
#if   defined(_WIN32)
#include <Windows.h>
#else // if defined(unix)
#include <pthread.h>
#include <ctime>
#endif

#if   defined(_WIN32)
class EvenementWin32
   : public IEvenement
{
   HANDLE evenement_;
public:
   EvenementWin32(ReinitialisationAutomatique)
      : evenement_(CreateEvent(0, FALSE, FALSE, 0))
   {
   }
   EvenementWin32(ReinitialisationManuelle)
      : evenement_(CreateEvent(0, TRUE, FALSE, 0))
   {
   }
   ~EvenementWin32() noexcept
      { CloseHandle(evenement_); }
   bool attendre() const volatile
      { return attendre(INFINITE); }
   bool attendre(const Temps<Milliseconde> &ms) const volatile
      { return WaitForSingleObject(evenement_, static_cast<DWORD>(ms.valeur())) == WAIT_OBJECT_0; }
   void provoquer() const volatile
      { SetEvent(evenement_); }
   void reinitialiser() const volatile
      { ResetEvent(evenement_); }
};
typedef EvenementWin32 evenement_impl;
#else // if defined(unix)
//
// ICI: to validate as Windows Events have no direct POSIX equivalents
//
// ICI: use condition_variable instead!
//
class EvenementPOSIX
   : public IEvenement
{
   mutable pthread_cond_t evenement_;
   mutable pthread_mutex_t mutex_;
   bool auto_reset_;
public:
   EvenementPOSIX(ReinitialisationAutomatique)
      : auto_reset_(true)
   {
      int err =  pthread_cond_init(&evenement_, 0);
      assert(!err);
      err = pthread_mutex_init(&mutex_, 0);
      assert(!err);
   }
   EvenementPOSIX(ReinitialisationManuelle)
      : auto_reset_(false)
   {
      int err =  pthread_cond_init(&evenement_, 0);
      assert(!err);
      err = pthread_mutex_init(&mutex_, 0);
      assert(!err);
   }
   ~EvenementPOSIX() noexcept
   {
      int err =  pthread_cond_destroy(&evenement_);
      assert(!err);
      err = pthread_mutex_destroy(&mutex_);
      assert(!err);
   }
   bool attendre() const volatile
   {
      EvenementPOSIX &self = const_cast<EvenementPOSIX &>(*this);
      int err = pthread_cond_wait(&self.evenement_, &self.mutex_);
      assert(!err);
      return true;
   }
   bool attendre(const Temps<Milliseconde> &ms) const volatile
   {
      EvenementPOSIX &self = const_cast<EvenementPOSIX &>(*this);
      timespec ts = { 0 };
      clock_gettime(CLOCK_REALTIME, &ts);
      ts.tv_nsec += Temps<Nanoseconde>(ms).valeur();
      int err = pthread_cond_timedwait(&self.evenement_, &self.mutex_, &ts);
      assert(!err);
      return true;
   }
   void provoquer() const volatile
   {
      EvenementPOSIX &self = const_cast<EvenementPOSIX &>(*this);
      int err = auto_reset_?
         pthread_cond_signal(&self.evenement_) :
         pthread_cond_broadcast(&self.evenement_);
      assert(!err);
   }
   void reinitialiser() const volatile
      { }
};
typedef EvenementPOSIX evenement_impl;
#endif

const volatile IEvenement *Evenement::creer(ReinitialisationAutomatique categ)
   { return new evenement_impl(categ); }
const volatile IEvenement *Evenement::creer(ReinitialisationManuelle categ)
   { return new evenement_impl(categ); }
void Evenement::liberer(const volatile IEvenement *p) noexcept
   { delete p; }
