#ifndef EVENEMENT_H
#define EVENEMENT_H

#include "Temps.h"
#include "incopiable.h"
#include "transition_algorithm.h"
#include <cstdlib>
#include <cassert>

struct IEvenement
{
   class ReinitialisationAutomatique {};
   class ReinitialisationManuelle  {};
   virtual bool attendre() const volatile = 0;
   virtual bool attendre(const Temps<Milliseconde>&) const volatile = 0;
   virtual void provoquer() const volatile = 0;
   virtual void reinitialiser() const volatile = 0;
protected:
   friend class Evenement;
   virtual ~IEvenement() = default;
};

class Evenement
   : Incopiable
{
public:
   typedef IEvenement::ReinitialisationAutomatique ReinitialisationAutomatique;
   typedef IEvenement::ReinitialisationManuelle ReinitialisationManuelle;
private:
   const volatile IEvenement *evenement_;
   static const volatile IEvenement *creer(ReinitialisationAutomatique);
   static const volatile IEvenement *creer(ReinitialisationManuelle);
   static void liberer(const volatile IEvenement *) noexcept;
public:
   template <class Categorie>
      Evenement(Categorie categ)
          : evenement_(creer(categ))
      {
      }
   Evenement(Evenement &&autre)
      : evenement_(std::move(autre.evenement_))
   {
      autre.evenement_ = nullptr;
   }
   void swap(Evenement &autre)
      { std::swap(evenement_, autre.evenement_); }
   void swap(Evenement &&autre)
      { std::swap(evenement_, autre.evenement_); }
   Evenement& operator=(Evenement &&autre)
   {
      swap(*this);
      return *this;
   }
   ~Evenement() noexcept
      { liberer(evenement_); }
   bool attendre() const volatile
      { assert(evenement_); return evenement_->attendre(); }
   bool attendre(const Temps<Milliseconde>& ms) const volatile
      { assert(evenement_); return evenement_->attendre(ms); }
   void provoquer() const volatile
      { assert(evenement_); return evenement_->provoquer(); }
   void reinitialiser() const volatile
      { assert(evenement_); return evenement_->reinitialiser(); }
};

#endif
