#ifndef TEMPS_H
#define TEMPS_H

//
// Rfrentiels
//
class Seconde {};
class Milliseconde {}; // rfrentiel neutre choisi
class Microseconde {};
class Nanoseconde {};

//
// Traits
//

#include "maths_ext.h"
#include "locale_extension.h"
#include "referent_traits.h"
#include <string>
#include <chrono>

struct base_temps_traits
   : base_referent_traits
{
   using value_type = double;
   static constexpr value_type default_value()
      { return value_type{}; }
};

template <class R>
   struct base_temps_stdrep_traits
      : base_temps_traits
   {
      using std_rep = R;
      static std_rep to_standard_rep(value_type val)
      {
         return std_rep{static_cast<typename std_rep::rep>(val)};
      }
   };

template <>
   struct referent_traits<Seconde>
      : base_temps_stdrep_traits<std::chrono::seconds>
   {
      template <class C>
         static std::basic_string<C> nom()
            { return creer_nom<C>("sec"); }
      static constexpr value_type to_neutral (value_type val)
         { return val * static_cast<value_type>(1000); }
      static constexpr value_type from_neutral (value_type val)
         { return val / static_cast<value_type>(1000); }
   };
template <>
   struct referent_traits<Milliseconde>
      : base_temps_stdrep_traits<std::chrono::milliseconds>
   {
      template <class C>
         static std::basic_string<C> nom()
            { return creer_nom<C>("ms"); }
      static constexpr value_type to_neutral(value_type val)
         { return val; }
      static constexpr value_type from_neutral(value_type val)
         { return val; }
   };
template <>
   struct referent_traits<Microseconde>
      : base_temps_stdrep_traits<std::chrono::microseconds>
   {
      template <class C>
         static std::basic_string<C> nom()
            { return creer_nom<C>("&mu;s"); }
      static constexpr value_type to_neutral(value_type val)
         { return val / static_cast<value_type>(1000); }
      static constexpr value_type from_neutral(value_type val)
         { return val * static_cast<value_type>(1000); }
   };
template <>
   struct referent_traits<Nanoseconde>
      : base_temps_stdrep_traits<std::chrono::nanoseconds>
   {
      template <class C>
         static std::basic_string<C> nom()
            { return creer_nom<C>("ns"); }
      static constexpr value_type to_neutral(value_type val)
         { return val / static_cast<value_type>(1000000); }
      static constexpr value_type from_neutral(value_type val)
         { return val * static_cast<value_type>(1000000); }
   };

//
// Temps
//
#include "relation.h"
#include <utility> // std::swap
#include <cassert>

template <class R>
   class Temps
      : relation::equivalence<Temps<R>>,
        relation::ordonnancement<Temps<R>>
   {
   public:
      using referentiel =  R;
      using value_type = typename
         referent_traits<R>::value_type;
   private:
      value_type valeur_;
   public:
      constexpr Temps(value_type val = referent_traits<R>::default_value())
         : valeur_{val}
      {
      }
      template <class U>
         Temps(const Temps<U> &t)
            : valeur_{referent_cast<R, U>(t.valeur())}
         {
         }
      void swap(Temps &t)
      {
         using std::swap;
         swap(valeur_, t.valeur_);
      }
      template <class U>
         Temps& operator=(const Temps<U> &t)
         {
            Temps{t}.swap(*this);
            return *this;
         }
      constexpr value_type valeur() const
         { return valeur_; }
      /*constexpr*/ bool operator==(const Temps &t) const
         { return maths_ext::assez_proches(valeur(), t.valeur()); }
      constexpr bool operator< (const Temps &t) const
         { return valeur() < t.valeur(); }
      Temps & operator++()
      {
         ++valeur_;
         return *this;
      }
      Temps operator++(int)
      {
         Temps temp{*this};
         operator++();
         return temp;
      }
      Temps operator+(const Temps &t) const
         { return Temps{valeur() + t.valeur()}; }
      Temps operator-(const Temps &t) const
         { return Temps{valeur() - t.valeur()}; }
      template <class U>
         Temps operator*(const U ratio) const
            { return Temps{valeur() * ratio}; }
      template <class U>
         Temps operator/(const U ratio) const
         {
            assert(ratio);
            return Temps{valeur() / ratio};
         }
      value_type operator/(const Temps &autre) const
      {
         assert(autre.valeur());
         return valeur() / autre.valeur();
      }
      explicit operator typename referent_traits<R>::std_rep() const
      {
         return referent_traits<R>::to_standard_rep(valeur());
      }
   };

//
// ICI: should be constexpr; switch as soon as the compiler supports it
//
inline Temps<Seconde> operator "" _s(unsigned long long val)
   { return Temps<Seconde>{static_cast<base_temps_traits::value_type>(val)}; }
inline Temps<Seconde> operator "" _s(long double val)
   { return Temps<Seconde>{static_cast<base_temps_traits::value_type>(val)}; }
inline Temps<Milliseconde> operator "" _ms(unsigned long long val)
   { return Temps<Milliseconde>{static_cast<base_temps_traits::value_type>(val)}; }
inline Temps<Milliseconde> operator "" _ms(long double val)
   { return Temps<Milliseconde>{static_cast<base_temps_traits::value_type>(val)}; }
inline Temps<Microseconde> operator "" _us(unsigned long long val)
   { return Temps<Microseconde>{static_cast<base_temps_traits::value_type>(val)}; }
inline Temps<Microseconde> operator "" _us(long double val)
   { return Temps<Microseconde>{static_cast<base_temps_traits::value_type>(val)}; }
inline Temps<Nanoseconde> operator "" _ns(unsigned long long val)
   { return Temps<Nanoseconde>{static_cast<base_temps_traits::value_type>(val)}; }
inline Temps<Nanoseconde> operator "" _ns(long double val)
   { return Temps<Nanoseconde>{static_cast<base_temps_traits::value_type>(val)}; }

#include <iosfwd>

template <class C, class R>
   std::basic_ostream<C>& operator<<
      (std::basic_ostream<C> &os, const Temps<R> &t)
   {
      return os << t.valeur() << widen_char<C>(' ')
                << referent_traits<R>::template nom<C>();
   }

#endif

