#include "low_level_socket_layer.h"
#include "socket_errors.h"
#include "FamillesAdresse.h"
#include "FamillesProtocole.h"
#include "Port.h"
#include "Adresse.h"
#include <type_traits>

using std::is_same;

#if   defined(_WIN32)
#include <Winsock2.h>

static_assert(is_same<low_level_layer::socket_t, SOCKET>::value, "Type socket_t incorrect pour abstraire SOCKET");

auto low_level_layer::invalid_socket() noexcept -> socket_t
{
   return INVALID_SOCKET;
}
bool low_level_layer::is_invalid(socket_t sck) noexcept
{
   return sck == invalid_socket();
}
bool low_level_layer::is_error(int code) noexcept
{
   return code == SOCKET_ERROR;
}
int low_level_layer::last_error() noexcept
{
   return WSAGetLastError();
}
void low_level_layer::conclure(socket_t sck) noexcept
{
   shutdown(sck, SD_BOTH);
}
void low_level_layer::fermer(socket_t sck) noexcept
{
   closesocket(sck);
}
void low_level_layer::rendre_bloquant(socket_t sck) noexcept
{
   unsigned long mode = 0;
   ioctlsocket(sck, FIONBIO, &mode);
}
void low_level_layer::rendre_non_bloquant(socket_t sck) noexcept
{
   unsigned long mode = 1; // tout non zro fait
   ioctlsocket(sck, FIONBIO, &mode);
}
enum
{
   ERR_ALREADY = WSAEALREADY,
   ERR_WOULDBLOCK = WSAEWOULDBLOCK,
   ERR_ISCONN = WSAEISCONN
};


#else // if defined(unix)
#include <netdb.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <errno.h>

auto low_level_layer::invalid_socket() noexcept -> socket_t 
{
   return -1;
}
bool low_level_layer::is_invalid(socket_t sck) noexcept
{
   return sck < 0;
}
bool low_level_layer::is_error(int code) noexcept
{
   return code < 0;
}
int low_level_layer::last_error() noexcept
{
   return errno;
}
void low_level_layer::conclure(socket_t sck) noexcept
{
   shutdown(sck, SHUT_RDWR);
}
void low_level_layer::fermer(socket_t sck) noexcept
{
   close(sck);
}
void low_level_layer::rendre_bloquant(socket_t sck) noexcept
{
   unsigned long mode = 0;
   ioctl(sck, FIONBIO, &mode);
}
void low_level_layer::rendre_non_bloquant(socket_t sck) noexcept
{
   unsigned long mode = 1; // tout non zro fait
   ioctl(sck, FIONBIO, &mode);
}
enum
{
   ERR_ALREADY = EALREADY,
   ERR_WOULDBLOCK = EWOULDBLOCK,
   ERR_ISCONN = EISCONN
};

#endif


auto low_level_layer::accepter(socket_t srv, Port &port, Adresse &adresse) -> socket_t
{
   sockaddr_in homologue = { 0 };
   int tailleHomologue = sizeof(sockaddr_in);
   socket_t sck = accept(
      srv, reinterpret_cast<sockaddr *>(&homologue), &tailleHomologue
   );
   if (is_invalid(sck))
   {
      erreur_type err = last_error();
      throw erreur_acceptation{err};
   }
   port = Port::denormaliser(homologue.sin_port);
#if   defined(_WIN32)
   adresse = Adresse(
      homologue.sin_addr.S_un.S_un_b.s_b1,
      homologue.sin_addr.S_un.S_un_b.s_b2,
      homologue.sin_addr.S_un.S_un_b.s_b3,
      homologue.sin_addr.S_un.S_un_b.s_b4
   );
#else // if defined(unix)
   adresse = Adresse(
      inet_ntoa(homologue.sin_addr/*.in_addr*/) // ICI: ugh!
   );
#endif
   return sck;
}

auto low_level_layer::creer(FamilleAdresse fadr, FamilleProtocole fpro) -> socket_t
{
   socket_t sck = socket(GestionnaireFamilles::get().trouver(fadr), SOCK_STREAM, GestionnaireProtocoles::get().trouver(fpro));
   if (low_level_layer::is_invalid(sck))
   {
      erreur_type err = last_error();
      throw erreur_creation{err};
   }
   return sck;
}

void low_level_layer::lier(socket_t sck, const Port &port, FamilleAdresse fadr)
{
   constexpr int TAILLE_FILE_ATTENTE = 16;
   sockaddr_in sa = { 0 };
   sa.sin_family = GestionnaireFamilles::get().trouver(fadr);
   sa.sin_port = port.valeur(Normalisee{});
   if (low_level_layer::is_error(bind(sck, reinterpret_cast<const sockaddr*>(&sa), sizeof(sockaddr_in))))
   {
      erreur_type err = last_error();
      throw erreur_liaison{err};
   }
   if (low_level_layer::is_error(listen(sck, TAILLE_FILE_ATTENTE)))
   {
      erreur_type err = last_error();
      throw erreur_liaison{err};
   }
}

void low_level_layer::connecter(socket_t sck, const Port &port, FamilleAdresse fadr, const Adresse &adresse)
{
   sockaddr_in serveur = { 0 };
   serveur.sin_family = GestionnaireFamilles::get().trouver(fadr);
   auto aname = adresse.valeur();
   serveur.sin_addr.s_addr = inet_addr(aname.c_str());
   serveur.sin_port = port.valeur(Normalisee{});
   if (low_level_layer::is_error(
      connect(
         sck, reinterpret_cast<const sockaddr*>(&serveur), sizeof (sockaddr)
      )
   ))
   {
      erreur_type err = last_error();
      throw erreur_connexion{err};
   }
}

bool low_level_layer::ok_if_nonblocking(erreur_type err)
{
   return !err || err == ERR_WOULDBLOCK /* || err == WSANOTINITIALISED*/;
} // ICI: le 2e cas, 10093, est une patch... Vrifier pourquoi on l'a parfois

int low_level_layer::send(socket_t sck, const char *buf, int buflen, int opts)
{
   int n = ::send(sck, buf, buflen, opts);
   if (!n)
      throw fermeture_homologue{0, true};
   else if (is_error(n))
      throw socket_exception{n};
   return n;
}
int low_level_layer::recv(socket_t sck, char *buf, int bufcap, int opts)
{
   int n = ::recv(sck, buf, bufcap, opts);
   if (!n)
      throw fermeture_homologue{0, true};
   else if (is_error(n))
      throw socket_exception{n};
   return n;
}

