#include "low_level_socket_layer.h"
#include "socket_errors.h"
#include "socket_client.h"
#include "socket_conversation.h"
#include "FamillesAdresse.h"
#include "FamillesProtocole.h"
#include "end_point.h"
#include <memory>
#include <atomic>
using namespace std;

class socket_client::Impl
{
public:
   using socket_t = low_level_layer::socket_t;
private:
   friend class socket_client;
   socket_t socket_;
   Port port_;
   Adresse adresse_;
   atomic<bool> bloquant_;
   FamilleAdresse fadr;
   using ep_ptr_type = std::unique_ptr<end_point>;
   unique_ptr<end_point> ep;
public:
   void clear()
   {
      socket_ = {};
   }
   Impl(FamilleAdresse fadr, FamilleProtocole fpro)
      : socket_{low_level_layer::creer(fadr, fpro)}, fadr{fadr}, adresse_{Adresse::local()}
   {
   }
   Impl(FamilleAdresse fadr, FamilleProtocole fpro, const Port &port, const Adresse &adresse)
      try : socket_(low_level_layer::creer(fadr, fpro)), port_(port), adresse_(adresse) // g++ n'aime pas les accolades avec un try : ...
      {
         ep = make_unique<end_point>(adresse, port);
         low_level_layer::connecter(socket_, port, fadr, adresse);
      }
      catch (erreur_connexion&)
      {
         low_level_layer::fermer(socket_);
         throw;
      }
   void connecter(const Port &port, const Adresse &adresse)
      try
      {
         low_level_layer::connecter(socket_, port, fadr, adresse);
         port_ = port;
         adresse_ = adresse;
      }
      catch (erreur_connexion &errconn)
      {
         if (!ok_if_nonblocking(errconn.erreur))
            throw;
      }
   ~Impl()
   {
      if (socket_)
      {
         low_level_layer::conclure(socket_);
         low_level_layer::fermer(socket_);
      }
   }
   int send(const char *src, int n) const
   {
      return low_level_layer::send(socket_, src, n, 0);
   }
   int recv(char *dest, int cap) const
   {
      return low_level_layer::recv(socket_, dest, cap, 0);
   }
   bool est_bloquant() const
   {
      return bloquant_;
   }
   //
   // ICI: synchroniser?
   //
   void rendre_bloquant()
   {
      low_level_layer::rendre_bloquant(socket_);
      bloquant_ = true;
   }
   void rendre_non_bloquant()
   {
      low_level_layer::rendre_non_bloquant(socket_);
      bloquant_ = {};
   }
   end_point endpoint() const
   {
      if (!ep) throw no_end_point{};
      return *ep;
   }
};

bool socket_client::est_bloquant() const
{
   return impl->est_bloquant();
}
void socket_client::rendre_bloquant()
{
   impl->rendre_bloquant();
}
void socket_client::rendre_non_bloquant()
{
   impl->rendre_non_bloquant();
}

end_point socket_client::endpoint() const
{
   return impl->endpoint();
}


socket_client::socket_client(socket_client && autre)
   : impl(move(autre.impl))
{
}

socket_client::socket_client(const Port &port, const Adresse &adresse)
   : impl{new Impl{Inet, Tcp, port, adresse}}
{
}

socket_conversation socket_client::extract_representation()
{
   socket_conversation conv{impl->socket_, impl->port_, impl->adresse_};
   impl->clear();
   return move(conv);
}

socket_client::~socket_client() = default;

int socket_client::envoyerPrimitif(const char *seq, int n)
{
   int nenv = impl->send(seq, n);
   if (!nenv || low_level_layer::is_error(nenv))
   {
      auto err = low_level_layer::last_error();
      if (est_bloquant() || !::ok_if_nonblocking(err))
         throw fermeture_homologue{err, low_level_layer::is_error(nenv)};
   }
   return nenv;
}

int socket_client::recevoirPrimitif(char *dest, std::size_t n)
{
   int nrec = impl->recv(dest, static_cast<int>(n));
   if (!nrec || low_level_layer::is_error(nrec))
   {
      auto err = low_level_layer::last_error();
      if (est_bloquant() || !::ok_if_nonblocking(err))
         throw fermeture_homologue{err, low_level_layer::is_error(nrec)};
   }
   return nrec;
}
