#include "socket_conversation.h"
#include "low_level_socket_layer.h"
#include "FamillesAdresse.h"
#include "FamillesProtocole.h"
#include "end_point.h"
#include "Adresse.h"
#include "incopiable.h"
#include "transition_algorithm.h"
#include "socket_errors.h"
#include <cstdlib>
#include <atomic>
#include <utility>
#include <memory>
using namespace std;

class socket_conversation::Impl
{
public:
   using socket_t = low_level_layer::socket_t;
private:
   socket_t socket_;
   atomic<bool> bloquant_;
   end_point ep;
public:
   Impl(socket_t sck, const Port &port, const Adresse &adresse)
      : socket_{sck}, bloquant_{}, ep{adresse, port}
   {
      rendre_bloquant();
   }
   ~Impl()
   {
      low_level_layer::conclure(socket_);
      low_level_layer::fermer(socket_);
   }
   int send(const char *src, int n) const
   {
      return low_level_layer::send(socket_, src, n, 0);
   }
   int recv(char *dest, int cap) const
   {
      return low_level_layer::recv(socket_, dest, cap, 0);
   }
   bool est_bloquant() const
   {
      return bloquant_;
   }
   //
   // ICI: synchroniser?
   //
   void rendre_bloquant()
   {
      low_level_layer::rendre_bloquant(socket_);
      bloquant_ = true;
   }
   void rendre_non_bloquant()
   {
      low_level_layer::rendre_non_bloquant(socket_);
      bloquant_ = {};
   }
   end_point endpoint() const
      { return ep; }
};

bool socket_conversation::est_bloquant() const
{
   return impl->est_bloquant();
}
void socket_conversation::rendre_bloquant()
{
   impl->rendre_bloquant();
}
void socket_conversation::rendre_non_bloquant()
{
   impl->rendre_non_bloquant();
}

end_point socket_conversation::endpoint() const
{
   return impl->endpoint();
}


socket_conversation::socket_conversation(socket_conversation && original)
   : impl{move(original.impl)}
{
}

socket_conversation::socket_conversation(socket_t socket, const Port &port, const Adresse &adresse)
   try : impl{new Impl{socket, port, adresse}}
   {
   }
   catch (...)
   {
      low_level_layer::fermer(socket);
      throw;
   }

socket_conversation::~socket_conversation() = default;

int socket_conversation::envoyerPrimitif(const char *seq, int n)
{
   int nenv = impl->send(seq, n);
   if (!nenv || low_level_layer::is_error(nenv))
   {
      auto err = low_level_layer::last_error();
      if (est_bloquant() || !::ok_if_nonblocking(err))
         throw fermeture_homologue{err, low_level_layer::is_error(nenv)};
   }
   return nenv;
}

int socket_conversation::recevoirPrimitif(char *dest, std::size_t n)
{
   int nrec = impl->recv(dest, static_cast<int>(n));
   if (!nrec || low_level_layer::is_error(nrec))
   {
      auto err = low_level_layer::last_error();
      if (est_bloquant() || !::ok_if_nonblocking(err))
         throw fermeture_homologue{err, low_level_layer::is_error(nrec)};
   }
   return nrec;
}

void socket_conversation::fermer()
{
   impl.reset();
}
