#ifndef TEXT_UTILS_H
#define TEXT_UTILS_H

#include "locale_extension.h"
#include "not_well_formed.h"
#include "func_composition.h"
#include "transition_algorithm.h"
#include <cstdlib>
#include <vector>
#include <deque>
#include <string>
#include <locale>
#include <cassert>

template <class C>
   C first_element(const std::basic_string<C> &s)
   {
      assert(!s.empty());
      return s.front();
   }

template <class C>
   C& first_element(std::basic_string<C> &s)
   {
      assert(!s.empty());
      return s.front();
   }

template <class C, int N>
   constexpr C first_element(const C (&s)[N])
   {
      return s[0];
   }

template <class C, int N>
   constexpr C& first_element(C (&s)[N])
   {
      return s[0];
   }

class empty_sequence {};

template <class C>
   constexpr C first_element(const C *s)
   {
      return s && s[0] ? s[0] : throw empty_sequence{};
   }

template <class C>
   constexpr C& first_element(C *s)
   {
      return s && s[0] ? s[0] : throw empty_sequence{};
   }

template <class C>
   C last_element(const std::basic_string<C> &s)
   {
      assert(!s.empty());
      return s.back();
   }

template <class C>
   C& last_element(std::basic_string<C> &s)
   {
      assert(!s.empty());
      return s.back();
   }

template <class C>
   std::basic_string<C> ltrim(std::basic_string<C> s)
   {
      using namespace std;
      auto &loc = chosen_locale();
      auto pred = [&](C c) { return !isspace(c, loc); };
      return s.substr(
         distance(begin(s), find_if(begin(s), end(s), pred))
      );
   }

template<class C>
   std::basic_string<C> rtrim(std::basic_string<C> s)
   {
      using namespace std;
      using str_type = basic_string<C>;
      using size_type = typename str_type::size_type;
      auto &loc = chosen_locale();
      auto pred = [&](C c) { return !isspace(c, loc); };
      return s.substr(
         size_type{},
         s.size() - distance(s.rbegin(), find_if(s.rbegin(), s.rend(), pred))
      );
   }

template <class C>
   std::basic_string<C> trim(std::basic_string<C> s)
   {
      using namespace std;
      auto &loc = chosen_locale();
      auto pred = [&](C c) { return !isspace(c, loc); };
      auto ndeb =
         distance(begin(s), find_if(begin(s), end(s), pred));
      auto nfin =
         distance(s.rbegin(), find_if(s.rbegin(), s.rend(), pred));
      return (ndeb || nfin)? s.substr(ndeb, s.size() - (ndeb + nfin)) : s;
   }

template <class C, class Pred>
   std::vector<std::basic_string<C> >
      split(const std::basic_string<C> &src, Pred crit)
   {
      using namespace std;
      using str_type = basic_string<C>;
      vector<str_type> v;
      auto p = begin(src);
      for (auto q = find_if(p, end(src), crit); q != end(src); q = find_if(++p, end(src), crit))
      {
         v.emplace_back(p, q);
         p = q;
      }
      if (p != end(src))
         v.emplace_back(p, end(src));
      return v;
   }

template <class C>
   std::vector<std::basic_string<C> >
      split(const std::basic_string<C> &src, C delim)
   {
      return split(src, [delim](C c) { return c == delim; });
   }

template <class C>
   std::basic_string<C>
      left_padding(
         std::basic_string<C> s, C sym,
         const typename std::basic_string<C>::size_type width
      )
   {
      if (s.size() < width)
         s.insert(begin(s), width - s.size(), sym);
      return s;
   }

template <class C>
   std::basic_string<C> left_zero_padding(
      const std::basic_string<C> &s,
      const typename std::basic_string<C>::size_type width
   )
      { return left_padding(s, widen_char<C>('0'), width); }

template <class>
   struct quote_impl;
template <>
   struct quote_impl<char>
   {
      static constexpr char symbol() { return '\"'; }
      static constexpr const char* str() { return "\""; }
   };
template <>
   struct quote_impl<wchar_t>
   {
      static constexpr wchar_t symbol() { return L'\"'; }
      static constexpr const wchar_t* str() { return L"\""; }
   };

template<class C>
   constexpr C quote_symbol()
   {
      return quote_impl<C>::symbol();
   }
template<class C>
   constexpr const C* quote_string()
   {
      return quote_impl<C>::str();
   }

template <template <class, class> class V = std::vector, template <class> class A = std::allocator, class C>
   V<std::basic_string<C>, A<std::basic_string<C>>>
      split_string_sensitive(
         const std::basic_string<C> &str, C delim,
         const std::basic_string<C> &string_delim = quote_string<C>()
      )
   {
      using str_type = std::basic_string<C>;
      V<str_type, A<str_type>> res;
      str_type cur;
      bool enquoting = {};
      for (auto c : str)
         if (enquoting)
         {
            cur += c;
            if (string_delim.find(c) != str_type::npos)
               enquoting = {};
         }
         else if (c == delim)
         {
            res.emplace_back(cur);
            cur.clear();
         }
         else if (string_delim.find(c) != str_type::npos)
         {
            cur += c;
            enquoting = true;
         }
         else
            cur += c;
      if (enquoting)
         throw not_well_formed(str);
      if (!cur.empty())
         res.emplace_back(cur);
      return res;
   }

class quote_unbalanced {};

template <class C>
   std::basic_string<C> strip_surrounding_quotes(std::basic_string<C> s, C quote = quote_symbol<C>())
   {
      using namespace std;
      using str_type = basic_string<C>;
      if (s.size() < 2) return s;
      if (s.front() == quote != s.back() == quote)
         throw quote_unbalanced{};
      return s.front() == quote ? str_type{next(begin(s)), prev(end(s))} : s;
   }

struct equal_quote_insensitive
{
   template <class C>
      bool operator()(const std::basic_string<C> &s0, const std::basic_string<C> &s1)
         { return strip_surrounding_quotes(s0) == strip_surrounding_quotes(s1); }
};

template <class C>
   std::basic_string<C>
      surround_with(std::basic_string<C> s, std::basic_string<C> beg, std::basic_string<C> end)
         { return beg + s + end; }

template <class C>
   std::basic_string<C>
      surround_with(std::basic_string<C> s, const C *beg, const C *end)
         { return beg + s + end; }

template <class C>
   std::basic_string<C>
      surround_with(std::basic_string<C> s, std::basic_string<C> surr)
         { return surround_with(std::move(s), surr, surr); }
template <class C>
   std::basic_string<C>
      surround_with(std::basic_string<C> s, const C *surr)
         { return surround_with(std::move(s), surr, surr); }

template <class C>
   std::basic_string<C> enquote(std::basic_string<C> s)
      { return surround_with(std::move(s), quote_string<C>()); }

template <class C, int N>
   std::basic_string<C> enquote(const C (&s)[N])
      { return enquote(static_cast<std::basic_string<C>>(s)); }

struct is_quote
{
   template <class C>
      constexpr bool operator()(C c)
      {
         return c == quote_symbol<C>();
      }
};

template <class S>
   bool enquoted(S &&s)
   {
      if (s.size() < 2) return false;
      return is_quote{}(s.front()) && is_quote{}(s.back());
   }

template <class C>
   bool starts_with(const std::basic_string<C> &s, const std::basic_string<C> &prefix)
   {
      if (s.size() <= prefix.size()) return false;
      using namespace std;
      return equal(begin(s), begin(s) + prefix.size(), begin(prefix));
      // ICI: changed to a compiler bug
      //using namespace std;
      //return s.size() > prefix.size() &&
      //       equal(begin(s), begin(s) + prefix.size(), begin(prefix));
   }
template <class C>
   bool starts_with(const std::basic_string<C> &s, const C *prefix)
   {
      return starts_with(s, std::basic_string<C>{prefix}); // ICI: optimiser!
   }

template <class C, class T>
   void make_message_impl(std::basic_stringstream<C> & sstr, T && arg)
   {
      static const auto SPACE = widen_char<C>(' '); // ICI: should I let the caller worry about space?
      sstr << arg << SPACE;
   }
template <class C, class T, class ...Ts>
   void make_message_impl(std::basic_stringstream<C> & sstr, T && arg, Ts && ... args)
   {
      make_message_impl<C>(sstr, std::forward<T>(arg));
      make_message_impl<C>(sstr, std::forward<Ts>(args)...);
   }
template <class C, class ...Ts>
   std::basic_string<C> make_message(Ts && ... args)
   {
      std::basic_stringstream<C> sstr;
      make_message_impl<C>(sstr, std::forward<Ts>(args)...);
      return sstr.str();
   }

#endif
